<?php
/*
Plugin Name: Mi Gestor de Menús (Simplificado)
Description: Plugin para gestionar múltiples menús con una interfaz simplificada de campos repetibles.
Version: 2.7
Author: Hernán De Vicente Fraile
Author URI: https://hernandvf.es
License: GPL2
Text Domain: mi-gestor-de-menus
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Salir si se accede directamente
}

class MiGestorDeMenusSimplificado {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_custom_admin_menu_page' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_action( 'admin_post_save_my_menus_data', array( $this, 'handle_save_my_menus_data' ) );
        add_action( 'admin_notices', array( $this, 'admin_notices' ) );

        // Shortcodes para el frontend
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_public_assets' ) );
        add_shortcode( 'menu_semanal', array( $this, 'render_weekly_menu_shortcode' ) );
        add_shortcode( 'menu_diario', array( $this, 'render_daily_menu_shortcode' ) );
        add_shortcode( 'carta_platos', array( $this, 'render_a_la_carte_menu_shortcode' ) );
        add_shortcode( 'carta_vinos', array( $this, 'render_wine_menu_shortcode' ) );
        add_shortcode( 'menu_especial', array( $this, 'render_special_menu_shortcode' ) );
    }

    // --- 1. AÑADIR PÁGINA DE ADMINISTRACIÓN PRINCIPAL ---
    public function add_custom_admin_menu_page() {
        add_menu_page(
            __( 'Gestor de Menús', 'mi-gestor-de-menus' ),
            __( 'Gestor de Menús', 'mi-gestor-de-menus' ),
            'manage_options',
            'my-menu-manager',
            array( $this, 'render_admin_menu_page' ),
            'dashicons-clipboard',
            20
        );
    }

    public function render_admin_menu_page() {
        $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'daily';
        ?>
        <div class="wrap">
            <h1><?php _e( 'Gestor de Menús del Restaurante', 'mi-gestor-de-menus' ); ?></h1>

            <?php settings_errors(); ?>

            <h2 class="nav-tab-wrapper">
                <a href="?page=my-menu-manager&tab=daily" class="nav-tab <?php echo ( $current_tab == 'daily' ) ? 'nav-tab-active' : ''; ?>"><?php _e( 'Menú Diario', 'mi-gestor-de-menus' ); ?></a>
                <a href="?page=my-menu-manager&tab=weekly" class="nav-tab <?php echo ( $current_tab == 'weekly' ) ? 'nav-tab-active' : ''; ?>"><?php _e( 'Menú Semanal', 'mi-gestor-de-menus' ); ?></a>
                <a href="?page=my-menu-manager&tab=a_la_carte" class="nav-tab <?php echo ( $current_tab == 'a_la_carte' ) ? 'nav-tab-active' : ''; ?>"><?php _e( 'Carta de Platos', 'mi-gestor-de-menus' ); ?></a>
                <a href="?page=my-menu-manager&tab=wine" class="nav-tab <?php echo ( $current_tab == 'wine' ) ? 'nav-tab-active' : ''; ?>"><?php _e( 'Carta de Vinos', 'mi-gestor-de-menus' ); ?></a>
                <a href="?page=my-menu-manager&tab=special" class="nav-tab <?php echo ( $current_tab == 'special' ) ? 'nav-tab-active' : ''; ?>"><?php _e( 'Menú Especial', 'mi-gestor-de-menus' ); ?></a>
            </h2>

            <div class="my-menu-manager-content">
                <form id="menu-manager-form" method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <input type="hidden" name="action" value="save_my_menus_data">
                    <input type="hidden" name="_wpnonce" value="<?php echo wp_create_nonce( 'my_menus_data_nonce' ); ?>">
                    <input type="hidden" name="current_tab" id="current-tab-input" value="<?php echo esc_attr( $current_tab ); ?>">
                    <input type="hidden" name="add_new_category_request" id="add-new-category-request" value="0">
                    <input type="hidden" name="add_new_items_request" id="add-new-items-request" value="">
                    <input type="hidden" name="add_new_category_name" id="add-new-category-name" value="">


                    <?php
                    // Mostrar los campos de opciones generales para cada pestaña
                    $this->render_general_options_fields( $current_tab );
                    ?>
                    <hr>
                    <?php
                    // Renderizar los campos repetibles para los platos/vinos
                    $this->render_menu_items_section( $current_tab );
                    ?>

                    <p class="submit">
                        <input type="submit" name="submit" id="submit" class="button button-primary" value="<?php _e( 'Guardar Cambios', 'mi-gestor-de-menus' ); ?>">
                    </p>
                </form>
            </div>
        </div>
        <?php
    }

    private function render_general_options_fields( $tab ) {
        // Datos comunes para los tipos de menú
        $common_fields = array(
            'daily' => array(
                'label' => 'Menú Diario',
                'price_label' => 'Precio General del Menú Diario:',
                'price_option' => 'daily_menu_general_price',
                'drinks_info_option' => 'daily_menu_drinks_info',
                'extra_info_option' => 'daily_menu_extra_info',
                'price_placeholder' => 'Ej: 15.50€',
            ),
            'weekly' => array(
                'label' => 'Menú Semanal',
                'price_label' => 'Precio General del Menú Semanal:',
                'price_option' => 'weekly_menu_general_price',
                'drinks_info_option' => 'weekly_menu_drinks_info',
                'extra_info_option' => 'weekly_menu_extra_info',
                'price_placeholder' => 'Ej: 22€',
            ),
            'special' => array(
                'label' => 'Menú Especial',
                'price_label' => 'Precio General del Menú Especial:',
                'price_option' => 'special_menu_general_price',
                'drinks_info_option' => 'special_menu_drinks_info',
                'extra_info_option' => 'special_menu_extra_info',
                'price_placeholder' => 'Ej: 30€',
            ),
            'a_la_carte' => array(
                'label' => 'Carta de Platos',
                'drinks_info_option' => 'a_la_carte_general_info',
            ),
            'wine' => array(
                'label' => 'Carta de Vinos',
                'drinks_info_option' => 'wine_general_info',
            ),
        );

        if ( isset( $common_fields[ $tab ] ) ) {
            $data = $common_fields[ $tab ];
            ?>
            <h2><?php printf( __( 'Opciones Generales del %s', 'mi-gestor-de-menus' ), $data['label'] ); ?></h2>
            <?php if ( isset( $data['price_option'] ) ) : ?>
                <label for="<?php echo esc_attr( $data['price_option'] ); ?>"><?php echo esc_html( $data['price_label'] ); ?></label>
                <input type="text" id="<?php echo esc_attr( $data['price_option'] ); ?>" name="<?php echo esc_attr( $data['price_option'] ); ?>" value="<?php echo esc_attr( get_option( $data['price_option'], '' ) ); ?>" class="regular-text" placeholder="<?php echo esc_attr( $data['price_placeholder'] ); ?>">
            <?php endif; ?>

            <?php if ( isset( $data['drinks_info_option'] ) ) : ?>
                <p class="description"><?php _e( 'Información de Pan, Bebidas o General (aparece al pie del menú):', 'mi-gestor-de-menus' ); ?></p>
                <textarea id="<?php echo esc_attr( $data['drinks_info_option'] ); ?>" name="<?php echo esc_attr( $data['drinks_info_option'] ); ?>" class="large-text" rows="3" placeholder="Ej: Incluye pan, bebida y postre."><?php echo esc_textarea( get_option( $data['drinks_info_option'], '' ) ); ?></textarea>
            <?php endif; ?>

            <?php if ( isset( $data['extra_info_option'] ) ) : ?>
                <p class="description"><?php _e( 'Información Extra (suplementos generales, etc.):', 'mi-gestor-de-menus' ); ?></p>
                <textarea id="<?php echo esc_attr( $data['extra_info_option'] ); ?>" name="<?php echo esc_attr( $data['extra_info_option'] ); ?>" class="large-text" rows="3" placeholder="Ej: Suplemento de terraza +1€."><?php echo esc_textarea( get_option( $data['extra_info_option'], '' ) ); ?></textarea>
            <?php endif; ?>
            <?php
        }
    }

    private function render_menu_items_section( $tab ) {
        $items_option_name = 'my_menu_items_' . $tab;
        $items_from_db = get_option( $items_option_name, array() );

        // Agrupar ítems por categoría para el renderizado
        $categorized_items_for_render = array();
        foreach ($items_from_db as $item) {
            $category_name = !empty($item['category']) ? $item['category'] : '';
            if (!empty($category_name)) {
                $categorized_items_for_render[$category_name][] = $item;
            }
        }
        // Conservar el orden de las categorías si ya existen en base a cómo se guardaron
        // Esto se logra al recorrer $items_from_db y construir $categorized_items_for_render
        // en el orden de aparición de las categorías en la DB.

        // --- Lógica para añadir nuevas categorías al recargar ---
        $new_category_name_from_request = isset($_GET['add_category']) ? sanitize_text_field($_GET['add_category']) : '';
        if ( !empty($new_category_name_from_request) ) {
            // Si la categoría no existe, la añadimos para que se renderice
            if (!array_key_exists($new_category_name_from_request, $categorized_items_for_render)) {
                $categorized_items_for_render[$new_category_name_from_request] = array();
            }
            // Asegurarse de que esta categoría tenga al menos 3 campos vacíos si se pidió añadir.
            // Esto es importante para que el usuario pueda empezar a escribir.
            for ($i = 0; $i < 3; $i++) {
                $categorized_items_for_render[$new_category_name_from_request][] = array();
            }
        }

        echo '<h3>' . __( 'Platos / Vinos', 'mi-gestor-de-menus' ) . '</h3>';
        echo '<p class="description">' . __( 'Arrastra para reordenar los platos dentro de cada sección. Haz clic en el icono "X" para eliminar un elemento o sección. Para añadir más platos o secciones, haz clic en el botón correspondiente y luego en "Guardar Cambios".', 'mi-gestor-de-menus' ) . '</p>';

        // Contenedor para las categorías
        echo '<div id="menu-categories-container">';

        // Si hay categorías guardadas o se ha añadido una nueva, renderizarlas
        if (!empty($categorized_items_for_render)) {
            foreach ($categorized_items_for_render as $category_name => $items_in_category) {
                // Lógica para añadir 3 items si se ha solicitado para ESTA categoría en particular
                $num_items_to_add = 0;
                $add_items_to_category_request = isset($_GET['add_items_to_category']) ? sanitize_text_field($_GET['add_items_to_category']) : '';
                if ( $add_items_to_category_request === $category_name ) {
                    $num_items_to_add = 3;
                }

                // Asegurar un mínimo de 3 items vacíos si la categoría está completamente vacía (sin nombre de plato)
                // y no se ha solicitado añadir items específicamente. Esto es para la carga inicial o si se eliminan todos.
                $has_named_items = false;
                foreach ($items_in_category as $item) {
                    if (!empty($item['name'])) {
                        $has_named_items = true;
                        break;
                    }
                }
                if (!$has_named_items && $num_items_to_add === 0) {
                    // Si no hay items con nombre Y no se ha pedido añadir, añade 3 por defecto
                    // Esto evita que una categoría se muestre vacía de inputs en el backend
                    $num_items_to_add = 3;
                }

                for ($i = 0; $i < $num_items_to_add; $i++) {
                    $items_in_category[] = array(); // Añadir items vacíos
                }

                $this->render_menu_category_block($tab, $category_name, $items_in_category);
            }
        } else {
            // Si no hay categorías guardadas y no se ha pedido añadir ninguna, añadir una por defecto al cargar la página
            if (empty($new_category_name_from_request) && !isset($_GET['add_items_to_category'])) {
                $default_category_name = ($tab === 'wine') ? 'Vinos Tintos' : 'Primeros Platos';
                $items_for_default_category = array_fill(0, 3, array()); // 3 items vacíos por defecto
                $this->render_menu_category_block($tab, $default_category_name, $items_for_default_category);
            }
        }

        echo '</div>'; // #menu-categories-container
        echo '<button type="button" class="button button-secondary add-new-category-btn">' . __( 'Añadir Nueva Sección / Categoría', 'mi-gestor-de-menus' ) . '</button>';

    }

    private function render_menu_category_block($tab, $category_name, $items_in_category) {
        // Genera un slug único para este renderizado de la categoría, útil para los names de los inputs
        // El slug con uniqid() se usa para que los grupos de inputs sean únicos en el POST,
        // pero category_original_name es la referencia estable para añadir y guardar.
        $category_block_id = sanitize_title($category_name) . '_' . uniqid();

        ?>
        <div class="menu-category-block" data-category-name="<?php echo esc_attr($category_name); ?>" data-category-block-id="<?php echo esc_attr($category_block_id); ?>">
            <div class="category-header">
                <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][category_name]" class="category-name-input large-text" value="<?php echo esc_attr($category_name); ?>" placeholder="<?php _e('Nombre de la Categoría (Ej: Primeros Platos)', 'mi-gestor-de-menus'); ?>">
                <input type="hidden" name="menu_items[<?php echo esc_attr($category_block_id); ?>][category_original_name]" value="<?php echo esc_attr($category_name); ?>" class="category-original-name-input">
                <button type="button" class="button button-secondary remove-category-block" title="<?php _e('Eliminar esta sección y todos sus platos', 'mi-gestor-de-menus'); ?>"><span class="dashicons dashicons-trash"></span></button>
            </div>
            <table class="wp-list-table widefat fixed striped items-table <?php echo ($tab === 'wine' ? 'wine-menu-table' : ''); ?>" data-category-block-id="<?php echo esc_attr($category_block_id); ?>">
                <thead>
                    <tr>
                        <th class="column-reorder"></th>
                        <th class="column-item-name"><?php _e( 'Nombre del Plato/Vino', 'mi-gestor-de-menus' ); ?></th>
                        <th class="column-description"><?php _e( 'Descripción', 'mi-gestor-de-menus' ); ?></th>
                        <?php if ( $tab === 'daily' || $tab === 'weekly' || $tab === 'special' ) : ?>
                            <th class="column-supplement"><?php _e( 'Suplemento', 'mi-gestor-de-menus' ); ?></th>
                        <?php elseif ( $tab === 'a_la_carte' ) : ?>
                            <th class="column-price"><?php _e( 'Precio', 'mi-gestor-de-menus' ); ?></th>
                        <? elseif ( $tab === 'wine' ) : ?>
                            <th class="column-bodega"><?php _e( 'Bodega', 'mi-gestor-de-menus' ); ?></th>
                            <th class="column-variedad"><?php _e( 'Variedad', 'mi-gestor-de-menus' ); ?></th>
                            <th class="column-price-copa"><?php _e( 'P. Copa', 'mi-gestor-de-menus' ); ?></th>
                            <th class="column-price-botella"><?php _e( 'P. Botella', 'mi-gestor-de-menus' ); ?></th>
                        <?php endif; ?>
                        <th class="column-delete"></th>
                    </tr>
                </thead>
                <tbody class="sortable-items-list">
                    <?php
                    // Si hay ítems en esta categoría, renderizarlos
                    if ( ! empty( $items_in_category ) ) {
                        foreach ( $items_in_category as $item ) {
                            $this->render_menu_item_row( $tab, $item, $category_block_id, $category_name );
                        }
                    }
                    ?>
                </tbody>
                <tfoot>
                    <tr>
                        <td colspan="<?php echo ($tab === 'wine' ? '8' : '5'); ?>">
                            <button type="button" class="button button-primary add-item-row-btn" data-category-name-for-add="<?php echo esc_attr($category_name); ?>"><?php _e( 'Añadir Plato / Vino', 'mi-gestor-de-menus' ); ?></button>
                        </td>
                    </tr>
                </tfoot>
            </table>
        </div>
        <?php
    }

    private function render_menu_item_row( $tab, $item = array(), $category_block_id = '', $category_original_name = '' ) {
        $item_name = isset( $item['name'] ) ? esc_attr( $item['name'] ) : '';
        $description = isset( $item['description'] ) ? esc_attr( $item['description'] ) : '';
        $supplement = isset( $item['supplement'] ) ? esc_attr( $item['supplement'] ) : '';
        $price = isset( $item['price'] ) ? esc_attr( $item['price'] ) : '';
        $bodega = isset( $item['bodega'] ) ? esc_attr( $item['bodega'] ) : '';
        $variedad = isset( $item['variedad'] ) ? esc_attr( $item['variedad'] ) : '';
        $price_copa = isset( $item['price_copa'] ) ? esc_attr( $item['price_copa'] ) : '';
        $price_botella = isset( $item['price_botella'] ) ? esc_attr( $item['price_botella'] ) : '';

        // Generar un ID único para cada fila. Si viene de la DB, usar su unique_id, sino generar uno nuevo.
        $row_id = isset($item['unique_id']) ? $item['unique_id'] : uniqid('item_');

        ?>
        <tr class="menu-item-row" data-id="<?php echo esc_attr($row_id); ?>">
            <td class="column-reorder"><span class="dashicons dashicons-move handle"></span></td>
            <td class="column-item-name">
                <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][name]" value="<?php echo $item_name; ?>" class="widefat" placeholder="<?php _e('Nombre', 'mi-gestor-de-menus'); ?>">
                <input type="hidden" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][category]" value="<?php echo esc_attr($category_original_name); ?>" class="item-category-input">
                <input type="hidden" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][unique_id]" value="<?php echo esc_attr($row_id); ?>" class="item-unique-id-input">
            </td>
            <td class="column-description">
                <textarea name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][description]" class="widefat" rows="2" placeholder="<?php _e('Descripción', 'mi-gestor-de-menus'); ?>"><?php echo $description; ?></textarea>
            </td>
            <?php if ( $tab === 'daily' || $tab === 'weekly' || $tab === 'special' ) : ?>
                <td class="column-supplement">
                    <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][supplement]" value="<?php echo $supplement; ?>" class="widefat" placeholder="<?php _e('+€', 'mi-gestor-de-menus'); ?>">
                </td>
            <?php elseif ( $tab === 'a_la_carte' ) : ?>
                <td class="column-price">
                    <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][price]" value="<?php echo $price; ?>" class="widefat" placeholder="<?php _e('€', 'mi-gestor-de-menus'); ?>">
                </td>
            <?php elseif ( $tab === 'wine' ) : ?>
                <td class="column-bodega">
                    <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][bodega]" value="<?php echo $bodega; ?>" class="widefat" placeholder="<?php _e('Bodega', 'mi-gestor-de-menus'); ?>">
                </td>
                <td class="column-variedad">
                    <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][variedad]" value="<?php echo $variedad; ?>" class="widefat" placeholder="<?php _e('Variedad', 'mi-gestor-de-menus'); ?>">
                </td>
                <td class="column-price-copa">
                    <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][price_copa]" value="<?php echo $price_copa; ?>" class="widefat" placeholder="<?php _e('€/copa', 'mi-gestor-de-menus'); ?>">
                </td>
                <td class="column-price-botella">
                    <input type="text" name="menu_items[<?php echo esc_attr($category_block_id); ?>][<?php echo esc_attr($row_id); ?>][price_botella]" value="<?php echo $price_botella; ?>" class="widefat" placeholder="<?php _e('€/botella', 'mi-gestor-de-menus'); ?>">
                </td>
            <?php endif; ?>
            <td class="column-delete">
                <button type="button" class="button button-small button-link-delete remove-item-row" title="<?php _e('Eliminar este plato/vino', 'mi-gestor-de-menus'); ?>"><span class="dashicons dashicons-trash"></span></button>
            </td>
        </tr>
        <?php
    }

    // --- 2. ENQUEUE SCRIPTS Y STYLES PARA EL ADMIN (AHORA CON ARCHIVO EXTERNO) ---
    public function enqueue_admin_assets( $hook_suffix ) {
        if ( 'toplevel_page_my-menu-manager' !== $hook_suffix ) {
             return; // Solo cargar en nuestra página
        }

        wp_add_inline_style( 'wp-admin', $this->get_admin_css() );

        wp_enqueue_script( 'jquery-ui-core' );
        wp_enqueue_script( 'jquery-ui-sortable' );

        $script_url = plugin_dir_url( __FILE__ ) . 'admin/js/my-menu-manager-admin-script.js';

        wp_enqueue_script(
            'my-menu-manager-admin-script',
            $script_url,
            array( 'jquery', 'jquery-ui-core', 'jquery-ui-sortable' ),
            filemtime( plugin_dir_path( __FILE__ ) . 'admin/js/my-menu-manager-admin-script.js' ), // Usar filemtime para la versión y evitar caché
            true // Cargar en el footer
        );
    }

    private function get_admin_css() {
        return '
            .my-menu-manager-content {
                background-color: #fff;
                padding: 20px;
                margin-top: 20px;
                border-radius: 5px;
                box-shadow: 0 1px 1px rgba(0,0,0,.04);
            }
            .nav-tab-wrapper {
                margin-bottom: 0;
            }
            .nav-tab {
                font-size: 16px;
                padding: 10px 15px;
                line-height: 20px;
            }
            .general-options-container label {
                font-weight: bold;
                display: block;
                margin-bottom: 5px;
            }
            .general-options-container input[type="text"],
            .general-options-container textarea {
                width: 100%;
                padding: 8px;
                border: 1px solid #ccc;
                border-radius: 4px;
                box-sizing: border-box;
                margin-bottom: 10px;
            }

            /* Estilos para la tabla de items */
            .items-table {
                margin-top: 20px;
            }
            .items-table thead th {
                white-space: nowrap;
            }
            .items-table input[type="text"],
            .items-table textarea {
                width: 100%;
                box-sizing: border-box;
                margin: 0;
                border: 1px solid #ddd;
                padding: 5px;
            }
            .items-table input[type="text"]:focus,
            .items-table textarea:focus {
                border-color: #5b9dd9;
                box-shadow: 0 0 0 1px #5b9dd9;
                outline: none;
            }
            .items-table .column-reorder {
                width: 30px;
                text-align: center;
                cursor: grab;
            }
            .items-table .handle {
                cursor: grab;
                color: #888;
            }
            .items-table .column-delete {
                width: 30px;
                text-align: center;
            }
            .items-table .remove-item-row .dashicons {
                color: #a00;
            }
            .items-table .remove-item-row:hover .dashicons {
                color: #f00;
            }

            /* Ancho de columnas específicos para vinos */
            .items-table.wine-menu-table .column-bodega,
            .items-table.wine-menu-table .column-variedad {
                width: 15%;
            }
            .items-table.wine-menu-table .column-price-copa,
            .items-table.wine-menu-table .column-price-botella {
                width: 10%;
            }
            .items-table.wine-menu-table .column-item-name {
                width: 20%;
            }
            .items-table.wine-menu-table .column-description {
                width: 25%;
            }

            /* Estilos para las categorías repetibles */
            .menu-category-block {
                border: 1px solid #ddd;
                padding: 15px;
                margin-bottom: 25px;
                background-color: #fdfdfd;
                border-radius: 5px;
                position: relative;
            }
            .menu-category-block .category-header {
                display: flex;
                align-items: center;
                margin-bottom: 15px;
            }
            .menu-category-block .category-name-input {
                flex-grow: 1;
                font-size: 1.2em;
                font-weight: bold;
                margin-right: 10px;
                background-color: #fff;
            }
            .menu-category-block .remove-category-block {
                color: #a00;
                border-color: #a00;
            }
            .menu-category-block .remove-category-block .dashicons {
                color: #a00;
            }
            .menu-category-block .remove-category-block:hover {
                background-color: #a00;
                color: #fff;
            }
            .menu-category-block .remove-category-block:hover .dashicons {
                color: #fff;
            }
            .add-new-category-btn, .add-item-row-btn {
                margin-top: 15px;
            }
        ';
    }

    // --- 3. MANEJAR EL GUARDADO DE DATOS ---
    public function handle_save_my_menus_data() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'No tienes permisos para hacer esto.', 'mi-gestor-de-menus' ) );
        }

        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'my_menus_data_nonce' ) ) {
            wp_die( __( 'Error de seguridad. Por favor, recarga la página e inténtalo de nuevo.', 'mi-gestor-de-menus' ) );
        }

        $current_tab = isset( $_POST['current_tab'] ) ? sanitize_text_field( $_POST['current_tab'] ) : 'daily';
        $message = 'success';

        // --- INICIO DE DEPURACIÓN EN PHP ---
        // error_log('DEBUG PHP: handle_save_my_menus_data iniciado para tab: ' . $current_tab);
        // error_log('DEBUG PHP: $_POST data: ' . print_r($_POST, true));
        // --- FIN DE DEPURACIÓN EN PHP ---

        // Guardar opciones generales
        $options_map = array(
            'daily' => array('daily_menu_general_price', 'daily_menu_drinks_info', 'daily_menu_extra_info'),
            'weekly' => array('weekly_menu_general_price', 'weekly_menu_drinks_info', 'weekly_menu_extra_info'),
            'special' => array('special_menu_general_price', 'special_menu_drinks_info', 'special_menu_extra_info'),
            'a_la_carte' => array('a_la_carte_general_info'),
            'wine' => array('wine_general_info'),
        );

        if ( isset( $options_map[$current_tab] ) ) {
            foreach ( $options_map[$current_tab] as $option_name ) {
                if ( isset( $_POST[$option_name] ) ) {
                    if ( strpos( $option_name, '_info' ) !== false ) { // Es un campo de textarea
                        update_option( $option_name, sanitize_textarea_field( $_POST[$option_name] ) );
                    } else {
                        update_option( $option_name, sanitize_text_field( $_POST[$option_name] ) );
                    }
                } else {
                     delete_option( $option_name ); // Si no se envía (vacío), eliminar
                }
            }
        }

        // Guardar items de menú
        $menu_items_to_save = array();
        if ( isset( $_POST['menu_items'] ) && is_array( $_POST['menu_items'] ) ) {
            foreach ( $_POST['menu_items'] as $category_block_id_from_post => $category_data ) {
                 // Usar category_name como nombre principal, si no está vacío, sino usar original
                $category_name = isset($category_data['category_name']) && !empty($category_data['category_name']) ? sanitize_text_field($category_data['category_name']) : (isset($category_data['category_original_name']) ? sanitize_text_field($category_data['category_original_name']) : '');

                // Si la categoría está vacía o el nombre es vacío, la ignoramos.
                if (empty($category_name)) {
                    continue;
                }

                foreach ( $category_data as $item_key => $item_data ) {
                    // Saltar los campos de nombre de categoría y cualquier cosa que no sea un array de item
                    if ( $item_key === 'category_name' || $item_key === 'category_original_name' || ! is_array( $item_data ) ) {
                        continue;
                    }

                    // Solo guardar si el campo de nombre del plato no está vacío
                    if ( empty( $item_data['name'] ) ) {
                        continue;
                    }

                    $sanitized_item = array();
                    // Asegúrate de que unique_id exista y sea válido, si no, genera uno nuevo
                    $sanitized_item['unique_id'] = isset($item_data['unique_id']) && !empty($item_data['unique_id']) ? sanitize_text_field( $item_data['unique_id'] ) : uniqid('item_');
                    $sanitized_item['name'] = sanitize_text_field( $item_data['name'] );
                    $sanitized_item['description'] = sanitize_textarea_field( $item_data['description'] );
                    $sanitized_item['category'] = $category_name; // Usamos el nombre (posiblemente actualizado)

                    if ( $current_tab === 'daily' || $current_tab === 'weekly' || $current_tab === 'special' ) {
                        $sanitized_item['supplement'] = sanitize_text_field( $item_data['supplement'] );
                    } elseif ( $current_tab === 'a_la_carte' ) {
                        $sanitized_item['price'] = sanitize_text_field( $item_data['price'] );
                    } elseif ( $current_tab === 'wine' ) {
                        $sanitized_item['bodega'] = sanitize_text_field( $item_data['bodega'] );
                        $sanitized_item['variedad'] = sanitize_text_field( $item_data['variedad'] );
                        $sanitized_item['price_copa'] = sanitize_text_field( $item_data['price_copa'] );
                        $sanitized_item['price_botella'] = sanitize_text_field( $item_data['price_botella'] );
                    }
                    $menu_items_to_save[] = $sanitized_item;
                }
            }
        }
        update_option( 'my_menu_items_' . $current_tab, $menu_items_to_save );
        // error_log('DEBUG PHP: Items finales guardados para ' . $current_tab . ': ' . print_r($menu_items_to_save, true));


        // --- Lógica para añadir más items/categorías después de guardar ---
        // Prepara los argumentos de redirección con la pestaña actual y el mensaje
        $redirect_args = array(
            'page'    => 'my-menu-manager',
            'tab'     => $current_tab,
            'message' => $message
        );

        // Si se solicitó añadir una nueva categoría
        if ( isset($_POST['add_new_category_request']) && $_POST['add_new_category_request'] === '1' ) {
            $category_name_for_add = sanitize_text_field($_POST['add_new_category_name']);
            if (!empty($category_name_for_add)) {
                $redirect_args['add_category'] = $category_name_for_add;
            }
        }
        // Si se solicitó añadir items a una categoría existente
        elseif ( isset($_POST['add_new_items_request']) && !empty($_POST['add_new_items_request']) ) {
            $redirect_args['add_items_to_category'] = sanitize_text_field($_POST['add_new_items_request']);
        }

        // Redirigir de forma segura
        wp_safe_redirect( add_query_arg( $redirect_args, admin_url( 'admin.php' ) ) );
        exit;
    }

    // --- 4. Mensajes de Administración ---
    public function admin_notices() {
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'my-menu-manager' ) {
            if ( isset( $_GET['message'] ) ) {
                if ( $_GET['message'] == 'success' ) {
                    echo '<div class="notice notice-success is-dismissible"><p>' . __( '¡Opciones guardadas correctamente!', 'mi-gestor-de-menus' ) . '</p></div>';
                } elseif ( $_GET['message'] == 'error' ) {
                    echo '<div class="notice notice-error is-dismissible"><p>' . __( 'Error al guardar las opciones. Por favor, inténtalo de nuevo.', 'mi-gestor-de-menus' ) . '</p></div>';
                }
            }
        }
    }

    // --- 5. ENQUEUE SCRIPTS Y STYLES PARA EL FRONTEND (INCRUSTADOS) ---
    public function enqueue_public_assets() {
        // Encola AOS CSS y JS desde CDN
        wp_enqueue_style( 'aos-css', 'https://unpkg.com/aos@2.3.1/dist/aos.css', array(), '2.3.1' );
        wp_enqueue_script( 'aos-js', 'https://unpkg.com/aos@2.3.1/dist/aos.js', array(), '2.3.1', true ); // true para cargar en el footer

        // Añade el CSS personalizado directamente
        wp_add_inline_style( 'aos-css', $this->get_public_css() ); // Depende de aos-css para cargar después

        // Añade el JS personalizado directamente
        wp_enqueue_script( 'my-menu-manager-public-inline-script', '', array( 'jquery', 'aos-js' ), '1.0', true );
        wp_add_inline_script( 'my-menu-manager-public-inline-script', $this->get_public_js() );
    }

    private function get_public_css() {
        return '
            /* --- Estilos CSS específicos para los Menús del Plugin --- */
            :root {
                --menu-primary-color: #C1262E;
                --menu-dark-text: #333;
                --menu-light-text: #666;
                --menu-bg-light: #f9f9f9;
                --menu-bg-white: #ffffff;
                --menu-border-color: #eee;
                --menu-shadow-light: rgba(0, 0, 0, 0.08);
                --menu-shadow-medium: rgba(0, 0, 0, 0.12);
                --menu-transition-speed: 0.3s;
            }

            .menu-page-container {
                max-width: 1200px;
                margin: 50px auto;
                padding: 30px 20px;
                font-family: \'Georgia\', serif;
                color: var(--menu-dark-text);
                line-height: 1.6;
                background-color: var(--menu-bg-white);
                border-radius: 15px;
                box-shadow: 0 10px 30px var(--menu-shadow-medium);
                overflow: hidden;
            }

            .menu-page-container .menu-header {
                text-align: center;
                margin-bottom: 60px;
                padding-bottom: 30px;
                border-bottom: 2px solid var(--menu-border-color);
            }

            .menu-page-container .menu-header h1 {
                font-size: 3.8em;
                color: var(--menu-primary-color);
                margin-bottom: 10px;
                text-transform: uppercase;
                letter-spacing: 2px;
                text-shadow: 2px 2px 4px rgba(0,0,0,0.1);
            }

            .menu-page-container .menu-subtitle {
                font-size: 1.5em;
                color: var(--menu-light-text);
                font-style: italic;
            }

            .menu-page-container .menu-sections {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
                gap: 40px;
                margin-bottom: 60px;
            }

            .menu-page-container .menu-category {
                background-color: var(--menu-bg-light);
                border-radius: 12px;
                padding: 30px;
                box-shadow: 0 5px 20px var(--menu-shadow-light);
                transition: transform var(--menu-transition-speed) ease-out, box-shadow var(--menu-transition-speed) ease-out;
            }

            .menu-page-container .menu-category:hover {
                transform: translateY(-8px);
                box-shadow: 0 12px 35px var(--menu-shadow-medium);
            }

            .menu-page-container .menu-category h2 {
                font-size: 2.5em;
                color: var(--menu-primary-color);
                text-align: center;
                margin-bottom: 30px;
                position: relative;
                padding-bottom: 10px;
            }

            .menu-page-container .menu-category h2::after {
                content: \'\';
                position: absolute;
                left: 50%;
                transform: translateX(-50%);
                bottom: 0;
                width: 60px;
                height: 3px;
                background-color: var(--menu-primary-color);
                border-radius: 2px;
            }

            .menu-page-container .menu-list {
                list-style: none;
                padding: 0;
                margin: 0;
            }

            .menu-page-container .menu-item {
                display: flex;
                flex-direction: column;
                padding: 15px 0;
                border-bottom: 1px dashed var(--menu-border-color);
                transition: background-color var(--menu-transition-speed) ease;
            }

            .menu-page-container .menu-item:last-child {
                border-bottom: none;
            }

            .menu-page-container .menu-item:hover {
                background-color: rgba(193, 38, 46, 0.05);
            }

            .menu-page-container .item-name {
                font-size: 1.3em;
                font-weight: bold;
                color: var(--menu-dark-text);
                margin-bottom: 5px;
                transition: color var(--menu-transition-speed) ease;
            }

            .menu-page-container .menu-item:hover .item-name {
                color: var(--menu-primary-color);
            }

            .menu-page-container .item-description {
                font-size: 0.95em;
                color: var(--menu-light-text);
                font-style: italic;
            }

            /* Precios generales para plato (Daily, Carta, Especial) */
            .menu-page-container .item-price-general {
                color: var(--menu-primary-color);
                font-weight: bold;
                font-size: 1.1em;
                margin-top: 5px;
                align-self: flex-end;
                background-color: rgba(193, 38, 46, 0.1);
                padding: 4px 8px;
                border-radius: 5px;
            }

            .menu-page-container .item-supplement {
                color: #555;
                font-weight: bold;
                font-size: 0.9em;
                margin-top: 5px;
                align-self: flex-end;
                background-color: rgba(0,0,0,0.05);
                padding: 2px 6px;
                border-radius: 3px;
            }
            .menu-page-container .item-price-with-supplement {
                display: flex;
                flex-direction: column;
                align-items: flex-end;
                margin-top: 5px;
            }


            /* Estilos específicos para Vinos */
            .menu-page-container .wine-details-container {
                display: flex;
                flex-wrap: wrap;
                align-items: center;
                justify-content: space-between;
                width: 100%;
                margin-top: 5px;
            }
            .menu-page-container .wine-bodega {
                font-weight: bold;
                margin-right: 10px;
                color: var(--menu-dark-text);
            }
            .menu-page-container .wine-variedad {
                color: var(--menu-light-text);
                font-style: italic;
                flex-grow: 1;
            }
            .menu-page-container .wine-prices {
                display: flex;
                gap: 15px;
                white-space: nowrap;
                margin-left: auto; /* Empuja los precios a la derecha */
            }
            .menu-page-container .price-copa,
            .menu-page-container .price-botella {
                background-color: rgba(193, 38, 46, 0.1);
                padding: 4px 8px;
                border-radius: 5px;
                color: var(--menu-primary-color);
                font-weight: bold;
                font-size: 1em;
            }


            .menu-page-container .menu-footer-info {
                text-align: center;
                padding-top: 40px;
                border-top: 2px solid var(--menu-border-color);
                background-color: var(--menu-bg-light);
                border-radius: 12px;
                box-shadow: 0 5px 20px var(--menu-shadow-light);
                padding: 30px;
            }

            .menu-page-container .drinks-info,
            .menu-page-container .extra-info {
                font-size: 1.1em;
                color: var(--menu-light-text);
                margin-bottom: 10px;
            }

            .menu-page-container .menu-price {
                font-size: 2.5em;
                font-weight: bold;
                color: var(--menu-primary-color);
                margin-top: 25px;
                text-shadow: 1px 1px 3px rgba(0,0,0,0.1);
            }

            /* --- Responsividad --- */
            @media (max-width: 992px) {
                .menu-page-container {
                    margin: 30px auto;
                    padding: 25px 15px;
                }
                .menu-page-container .menu-header h1 {
                    font-size: 3em;
                }
                .menu-page-container .menu-subtitle {
                    font-size: 1.3em;
                }
                .menu-page-container .menu-category h2 {
                    font-size: 2em;
                }
                .menu-page-container .item-name {
                    font-size: 1.2em;
                }
                .menu-page-container .item-description {
                    font-size: 0.9em;
                }
                .menu-page-container .wine-details-container {
                    flex-direction: column;
                    align-items: flex-start;
                }
                .menu-page-container .wine-prices {
                    margin-left: 0;
                    margin-top: 5px;
                }
            }

            @media (max-width: 768px) {
                .menu-page-container .menu-sections {
                    grid-template-columns: 1fr;
                    gap: 30px;
                }
                .menu-page-container .menu-header h1 {
                    font-size: 2.5em;
                }
                .menu-page-container .menu-subtitle {
                    font-size: 1.1em;
                }
                .menu-page-container .menu-category {
                    padding: 25px;
                }
                .menu-page-container .menu-category h2 {
                    font-size: 1.8em;
                }
                .menu-page-container .menu-item {
                    padding: 12px 0;
                }
                .menu-page-container .item-name {
                    font-size: 1.1em;
                }
                .menu-page-container .item-description {
                    font-size: 0.85em;
                }
                .menu-page-container .menu-footer-info {
                    padding: 25px;
                }
                .menu-page-container .drinks-info,
                .menu-page-container .extra-info {
                    font-size: 1em;
                }
                .menu-page-container .menu-price {
                    font-size: 1.8em;
                }
            }

            @media (max-width: 480px) {
                .menu-page-container {
                    padding: 20px 10px;
                }
                .menu-page-container .menu-header h1 {
                    font-size: 2em;
                    letter-spacing: 1px;
                }
                .menu-page-container .menu-subtitle {
                    font-size: 0.9em;
                }
                .menu-page-container .menu-category h2 {
                    font-size: 1.6em;
                }
                .menu-page-container .item-name {
                    font-size: 1em;
                }
                .menu-page-container .item-description {
                    font-size: 0.85em;
                }
                .menu-page-container .menu-price {
                    font-size: 1.8em;
                }
            }
        ';
    }

    private function get_public_js() {
        return '
            jQuery(document).ready(function($) {
                if (typeof AOS === \'object\' && typeof AOS.init !== \'undefined\' && !document.querySelector(\'body[data-aos-easing]\')) {
                    AOS.init({
                        duration: 1000,
                        once: true
                    });
                } else if (typeof AOS === \'object\' && typeof AOS.refresh === \'function\') {
                    AOS.refresh();
                }
            });
        ';
    }


    // --- 6. SHORTCODES PARA EL FRONTEND ---

    private function get_menu_items_html( $menu_type_slug, $title, $general_price_option_name = '', $drinks_info_option_name = '', $extra_info_option_name = '' ) {
        ob_start();

        // error_log('DEBUG FRONTEND: Renderizando shortcode para: ' . $menu_type_slug);

        $items_option_name = 'my_menu_items_' . $menu_type_slug;
        $all_items = get_option( $items_option_name, array() );

        // error_log('DEBUG FRONTEND: Opción de BD a buscar: ' . $items_option_name);
        // error_log('DEBUG FRONTEND: Items recuperados de la BD para ' . $menu_type_slug . ': ' . print_r($all_items, true));

        // Agrupar ítems por categoría. Usamos un array auxiliar para mantener el orden.
        $categorized_items = array();
        $category_order = array(); // Para mantener el orden de las categorías tal como se guardaron
        foreach ( $all_items as $item ) {
            $category = ! empty( $item['category'] ) ? $item['category'] : 'Sin Categoría';
            if (!in_array($category, $category_order)) {
                $category_order[] = $category;
            }
            $categorized_items[ $category ][] = $item;
        }

        $general_price = get_option( $general_price_option_name, '' );
        $drinks_info = get_option( $drinks_info_option_name, '' );
        $extra_info = get_option( $extra_info_option_name, '' );

        echo '<div class="menu-page-container ' . esc_attr($menu_type_slug) . '-container">';
            echo '<header class="menu-header">';
                echo '<h1>' . esc_html( $title ) . '</h1>';
                if ( ! empty( $general_price ) && $general_price_option_name != '' ) {
                    echo '<p class="menu-subtitle">Precio: ' . esc_html( $general_price ) . '</p>';
                }
            echo '</header>';

            if ( ! empty( $categorized_items ) ) {
                echo '<div class="menu-sections">';
                    $index = 0;
                    foreach ( $category_order as $category_name ) { // Iteramos por el orden guardado
                        $items_in_category = $categorized_items[$category_name];

                        // Solo renderizar categorías que tengan al menos un item con nombre
                        $has_actual_items = false;
                        foreach($items_in_category as $item) {
                            if(!empty($item['name'])) {
                                $has_actual_items = true;
                                break;
                            }
                        }
                        if (!$has_actual_items) {
                            continue; // No mostrar categorías completamente vacías en el frontend
                        }

                        echo '<div class="menu-category" data-aos="fade-up" data-aos-delay="' . esc_attr( $index * 100 ) . '">';
                            echo '<h2>' . esc_html( $category_name ) . '</h2>';
                            echo '<ul class="menu-list">';
                            foreach ( $items_in_category as $item ) {
                                // Solo renderizar en el frontend si el item tiene un nombre
                                if (empty($item['name'])) {
                                    continue;
                                }
                                echo '<li class="menu-item">';
                                    echo '<span class="item-name">' . esc_html( $item['name'] ) . '</span>';
                                    if ( ! empty( $item['description'] ) ) {
                                        echo '<span class="item-description">' . esc_html( $item['description'] ) . '</span>';
                                    }

                                    // Precios y suplementos específicos
                                    if ( ($menu_type_slug === 'daily' || $menu_type_slug === 'weekly' || $menu_type_slug === 'special') && ! empty( $item['supplement'] ) ) {
                                        echo '<span class="item-supplement">' . esc_html( $item['supplement'] ) . '</span>';
                                    } elseif ( $menu_type_slug === 'a_la_carte' && ! empty( $item['price'] ) ) {
                                        echo '<span class="item-price-general">' . esc_html( $item['price'] ) . '</span>';
                                    } elseif ( $menu_type_slug === 'wine' ) {
                                        echo '<div class="wine-details-container">';
                                            if ( ! empty( $item['bodega'] ) ) {
                                                echo '<span class="wine-bodega">' . esc_html( $item['bodega'] ) . '</span>';
                                            }
                                            if ( ! empty( $item['variedad'] ) ) {
                                                echo '<span class="wine-variedad">' . esc_html( $item['variedad'] ) . '</span>';
                                            }
                                            echo '<div class="wine-prices">';
                                                if ( ! empty( $item['price_copa'] ) ) {
                                                    echo '<span class="price-copa">Copa: ' . esc_html( $item['price_copa'] ) . '</span>';
                                                }
                                                if ( ! empty( $item['price_botella'] ) ) {
                                                    echo '<span class="price-botella">Botella: ' . esc_html( $item['price_botella'] ) . '</span>';
                                                }
                                            echo '</div>';
                                        echo '</div>';
                                    }
                                echo '</li>';
                            }
                            echo '</ul>';
                        echo '</div>';
                        $index++;
                    }
                echo '</div>'; // .menu-sections
            } else {
                echo '<p>' . __( 'No hay elementos de menú configurados para este tipo.', 'mi-gestor-de-menus' ) . '</p>';
            }

            echo '<div class="menu-footer-info" data-aos="fade-up" data-aos-delay="300">';
                if ( ! empty( $drinks_info ) ) {
                    echo '<p class="drinks-info">' . wpautop(wp_kses_post( $drinks_info )) . '</p>'; // wpautop para saltos de línea, wp_kses_post para HTML básico
                }
                if ( ! empty( $extra_info ) ) {
                    echo '<p class="extra-info">' . wpautop(wp_kses_post( $extra_info )) . '</p>';
                }
                if ( ! empty( $general_price ) && $general_price_option_name != '' ) {
                    echo '<p class="menu-price">' . esc_html( $general_price ) . '</p>';
                }
            echo '</div>'; // .menu-footer-info
        echo '</div>'; // .menu-page-container

        return ob_get_clean();
    }

    public function render_weekly_menu_shortcode( $atts ) {
        return $this->get_menu_items_html(
            'weekly',
            __( 'Menú Semanal', 'mi-gestor-de-menus' ),
            'weekly_menu_general_price',
            'weekly_menu_drinks_info',
            'weekly_menu_extra_info'
        );
    }

    public function render_daily_menu_shortcode( $atts ) {
        return $this->get_menu_items_html(
            'daily',
            __( 'Menú Diario', 'mi-gestor-de-menus' ),
            'daily_menu_general_price',
            'daily_menu_drinks_info',
            'daily_menu_extra_info'
        );
    }

    public function render_a_la_carte_menu_shortcode( $atts ) {
        return $this->get_menu_items_html(
            'a_la_carte',
            __( 'Carta de Platos', 'mi-gestor-de-menus' ),
            '', // No hay precio general de opción
            'a_la_carte_general_info',
            ''
        );
    }

    public function render_wine_menu_shortcode( $atts ) {
        return $this->get_menu_items_html(
            'wine',
            __( 'Carta de Vinos', 'mi-gestor-de-menus' ),
            '', // No hay precio general de opción
            'wine_general_info',
            ''
        );
    }

    public function render_special_menu_shortcode( $atts ) {
        return $this->get_menu_items_html(
            'special',
            __( 'Menú Especial', 'mi-gestor-de-menus' ),
            'special_menu_general_price',
            'special_menu_drinks_info',
            'special_menu_extra_info'
        );
    }
}

// Instanciar la clase del plugin
new MiGestorDeMenusSimplificado();